"""
Loop fluctuation correlation simulation package.

This package wraps the core scripts from the standalone loop‑fluctuation
simulation (see the upstream vol4‑loop‑fluctuation‑sim repository) into a
Python package.  It exposes helper functions for generating tick‑flip
counts, sampling gauge‑field configurations, running correlation analysis
and generating a summary report.  A high level entry point is provided
in :mod:`vol4_loop_fluctuation_sim.run`.
"""

from importlib import import_module

__all__ = [
    "generate_flip_counts",
    "sample_gauge_fields",
    "run_correlation",
    "generate_report",
]


def __getattr__(name):
    """Dynamically import top‑level helper functions from the scripts package.

    The standalone scripts live in the ``scripts`` subpackage and define
    free functions such as :func:`generate_flip_counts` in
    ``generate_flip_counts.py``.  This function proxies attribute access to
    the corresponding module so that callers can simply import
    ``vol4_loop_fluctuation_sim.generate_flip_counts``.
    """
    if name in __all__:
        module = import_module(f"vol4_loop_fluctuation_sim.scripts.{name}")
        return getattr(module, name)
    raise AttributeError(f"module {__name__!r} has no attribute {name!r}")